// background.js
chrome.runtime.onInstalled.addListener(async () => {
  await initializeExtensionState();
});

// Inicializar el estado de la extensión
async function initializeExtensionState() {
  const { enabled } = await chrome.storage.sync.get("enabled");
  if (enabled === undefined) {
    await chrome.storage.sync.set({ enabled: true }); // Activar por defecto
    applyRules(true);
  } else {
    applyRules(enabled);
  }
}

// Aplicar o eliminar reglas dinámicas
async function applyRules(enable) {
  chrome.declarativeNetRequest.getDynamicRules({}, async (existingRules) => {
    const existingRuleIds = new Set(existingRules.map((rule) => rule.id));
    let nextId = 1; // Iniciar el próximo ID disponible

    if (enable) {
      try {
        console.log("Obteniendo blocklist y whitelist...");

        // Obtener la lista de bloqueo
        const blocklistResponse = await fetch("https://putosanuncios.fun/blocklist.json");
        if (!blocklistResponse.ok) throw new Error("No se pudo obtener el archivo de dominios bloqueados.");
        const blocklist = await blocklistResponse.json();
        console.log("Dominios bloqueados obtenidos:", blocklist);

        // Obtener la lista blanca
        const whitelistResponse = await fetch("https://putosanuncios.fun/whitelist.json");
        if (!whitelistResponse.ok) throw new Error("No se pudo obtener el archivo de dominios permitidos.");
        const whitelist = await whitelistResponse.json();
        console.log("Dominios permitidos obtenidos:", whitelist);

        // Crear reglas para la lista blanca (permitir)
        const allowRules = whitelist.map((domain) => {
          while (existingRuleIds.has(nextId)) nextId++; // Asegurar ID único
          const rule = {
            id: nextId,
            priority: 2, // Prioridad más alta
            action: { type: "allow" },
            condition: {
              urlFilter: domain,
              resourceTypes: [
                "main_frame",
                "sub_frame",
                "script",
                "image",
                "stylesheet",
                "xmlhttprequest",
              ],
            },
          };
          existingRuleIds.add(nextId); // Registrar ID
          return rule;
        });

        // Filtrar los dominios bloqueados que no estén en la lista blanca
        const filteredBlocklist = blocklist.filter(domain => !whitelist.includes(domain));
        console.log("Dominios a bloquear después del filtrado:", filteredBlocklist);

        const blockRules = filteredBlocklist.map((domain) => {
          while (existingRuleIds.has(nextId)) nextId++; // Asegurar ID único
          const rule = {
            id: nextId,
            priority: 1, // Prioridad más baja
            action: { type: "block" },
            condition: {
              urlFilter: domain,
              resourceTypes: [
                "main_frame",
                "sub_frame",
                "script",
                "image",
                "stylesheet",
                "xmlhttprequest",
              ],
            },
          };
          existingRuleIds.add(nextId); // Registrar ID
          return rule;
        });

        // Actualizar las reglas dinámicas
        chrome.declarativeNetRequest.updateDynamicRules(
          {
            removeRuleIds: Array.from(existingRuleIds),
            addRules: [...allowRules, ...blockRules],
          },
          () => {
            console.log("Reglas actualizadas correctamente: Permitir y bloquear aplicadas.");
          }
        );
      } catch (error) {
        console.error("Error al descargar o aplicar las reglas:", error);
      }
    } else {
      chrome.declarativeNetRequest.updateDynamicRules(
        { removeRuleIds: Array.from(existingRuleIds) },
        () => {
          console.log("Reglas desactivadas.");
        }
      );
    }
  });
}

// Listener para cambios en el almacenamiento
chrome.storage.onChanged.addListener((changes) => {
  if (changes.enabled) {
    applyRules(changes.enabled.newValue);
  }
});
